#!/bin/bash
##################################################################
# Filename: AutoInit_GBase8s.sh
# Function: Auto install GBase 8s software and auto init database.
# Write by: liaosnet@gbasedbt.com
# Version : 1.4.17   update date: 2026-01-09
##################################################################
##### Defind env
export LANG=C
_loginfo(){
  echo -e "[$(date +'%Y-%m-%d %H:%M:%S')] $*"
}
##### Get Parameter
while [[ $# -gt 0 ]]
do
  key="$1"
  case $key in
    -d) DATADIR="$2";     shift 2 ;;
    -i) INSTALL_DIR="$2"; shift 2 ;;
    -p) USER_HOME="$2";   shift 2 ;;
    -s) DBS1GB="$2";      shift 2 ;;
    -l) GBASELOCALE="$2"; shift 2 ;;
    -o) SOFTONLY="$2";    shift 2 ;;
    -u) USER_NAME="$2";   shift 2 ;;
    -n) USER_SERV="$2";   shift 2 ;;
    -c) USER_CPU="$2";    shift 2 ;;
    -m) USER_MEM="$2";    shift 2 ;;
    -t) USER_INST="$2";   shift 2 ;;
    -a) SECNEED="$2";     shift 2 ;;
    -E) ERTYPE="$2";      shift 2 ;;
    -T) HACTYPE="$2";     shift 2 ;;
    -H1) IP_PRI="$2";     shift 2 ;;
    -H2) IP_SEC="$2";     shift 2 ;;	
    *)
        cat <<EOF
Usage:
    AutoInit_GBase8s.sh [-d path] [-i path] [-p path] [-s y|n] [-l locale] [-u user] [-o y|n] 
                        [-n servername] [-c num_of_cpu] [-m num_of_memory] [-t type_of_instance]
                        [-a y|n]
                        [-E ertype] [-T hactype] [-H1 primary_ipaddr] [-H2 secondary_ipaddr]

        -d  path    The path of dbspace.
        -i  path    The path of install software.
        -p  path    The path of home path.
        -s  y|n     Value of dbspace is 1GB? Yes/No, default is N.
        -u  user    The user name for SYSDBA, gbasedbt/informix, default is gbasedbt
        -l  locale  DB_LOCALE/CLIENT_LOCALE/SERVER_LOCALE value.
        -o  y|n     Only install software? Yes/No, default is N.
        -n  NAME    Servername, default is gbase01.
        -c  NUM     Number of CPU use.
        -m  NUM     Number of MB Memory use.
        -t  TYPE    Type of instance will install, [small], if use this, ignore -c and -m. 
        -a  y|n     Security need, default N.
        -E  ERType  Type of ER, 0(NONE)/1(STD_PRI)/2(STD_SEC)/11(HAC_PRI)/12(HAC_SEC)
        -T  HACType Type of hac cluster, PRI/SEC/STD.
        -H1 ipaddr  Ipaddr for primary server.
        -H2 ipaddr  Ipaddr for secondary server.

EOF
        exit 1
        ;;
  esac
done
##### Define Parameter
USER_NAME=${USER_NAME:-gbasedbt}
if [ x"$USER_NAME" = "xgbasedbt" -o x"$USER_NAME" = "xinformix" ]; then
  _loginfo "The SYSDBA user is: $USER_NAME"
else
  echo "Username error!"
  exit 1
fi
USER_UPPER=$(echo "$USER_NAME"|tr "[a-z]" "[A-Z]")
USER_HOME=${USER_HOME:-/home/gbase}
USER_PASS=GBase123$%
INSTALL_DIR=${INSTALL_DIR:-/opt/gbase}
GBASESERVER=${USER_SERV:-gbase01}
GBASELOCALE=${GBASELOCALE:-zh_CN.utf8}
USER_CPU=${USER_CPU:-0}
USER_MEM=${USER_MEM:-0}
USER_INST=${USER_INST:-none}
PORTNO=9099
PORTPRIV=9088
DATADIR=${DATADIR:-/data/gbase}
DBS1GB=${DBS1GB:-n}
SECNEED=${SECNEED:-n}
ERTYPE=${ERTYPE:-0}
HACTYPE=${HACTYPE:-STD}
HACTYPE=$(echo "$HACTYPE"|tr "[a-z]" "[A-Z]")
IP_PRI=${IP_PRI:-0.0.0.0}
IP_SEC=${IP_SEC:-0.0.0.0}
SN_PRI=std01
SN_SEC=hac02
if [ ${ERTYPE:-0} -eq 1 ]; then
  HACTYPE=STD
elif [ ${ERTYPE:-0} -eq 2 ]; then
  HACTYPE=STD
  SN_PRI=std11
elif [ ${ERTYPE:-0} -eq 11 ]; then
  SN_PRI=hac01
  SN_SEC=hac02
  if [ x"${HACTYPE}" = xPRI ]; then 
    GBASESERVER=hac01
  else
    GBASESERVER=hac02
  fi
elif [ ${ERTYPE:-0} -eq 12 ]; then
  SN_PRI=hac11
  SN_SEC=hac12
  if [ x"${HACTYPE}" = xPRI ]; then 
    GBASESERVER=hac11
  else
    GBASESERVER=hac12
  fi
else
  if [ x"${HACTYPE}" = xPRI ]; then 
    GBASESERVER=hac01
    SN_PRI=hac01
  elif [ x"${HACTYPE}" = xSEC ]; then 
    GBASESERVER=hac02
    SN_PRI=hac01
    SN_SEC=hac02
  else
    GBASESERVER=std01
  fi
fi

TMP_USER_CPU=$(echo "${USER_CPU}" | sed 's/[0-9]//g')
if [ ! x"${TMP_USER_CPU}" = x ]; then
  USER_CPU=0
fi
TMP_USER_MEM=$(echo "${USER_MEM}" | sed 's/[0-9]//g')
if [ ! x"${TMP_USER_MEM}" = x ]; then
  USER_MEM=0
fi

WORKDIR=$(pwd)
##### Check env
if [ ! x"$(whoami)" = "xroot" ]; then
  echo "Must run as user: root"
  exit 1
fi 
if [ x"${USER_HOME}" = x"${INSTALL_DIR}" ]; then
  INSTALL_DIR=${USER_HOME}/gbase
fi
if [ -d ${INSTALL_DIR} ] && [ ! x"$(ls -A ${INSTALL_DIR})" = x ]; then
  INSTALL_DIR=${INSTALL_DIR}/Server
fi
if [ -d ${DATADIR} ] && [ ! x"$(ls -A ${DATADIR})" = x ]; then
  DATADIR=${INSTALL_DIR}/data
fi

SOFTPACKNAME=$(ls GBase*.tar 2>/dev/null)
if [ x"$SOFTPACKNAME" = x ]; then
  _loginfo "ERROR: GBase Database Soft Package not found."
  exit 2
fi
# check cpu arch & software 
# todo: other cpu
CPUTYPE=$(uname -m)
SOFTCPU=$(echo "${SOFTPACKNAME}" | awk '
    BEGIN{IGNORECASE=1}
    {if(/FT|KP9|aarch64/){print "aarch64";exit}
     else if(/SW/){print "sw64";exit}
     else if(/Loongson/){print "loongarch64";exit}
     else {print "x86_64"}
    }')	 

if [ x"${CPUTYPE}" = "xx86_64" -o x"${CPUTYPE}" = "xaarch64" ]; then 
  if [ ! x"${CPUTYPE}" = x"${SOFTCPU}" ]; then 
    _loginfo "ERROR: Current CPU type is: ${CPUTYPE}, BUT software package is: ${SOFTCPU} "
    exit 2
  fi
else
  _loginfo "Current CPU type is: ${CPUTYPE}, Software package is: ${SOFTCPU} "
fi							

### CheckEnv.sh
bash CheckEnv.sh
if [ $? -gt 0 ]; then
  exit 3
fi

# IP use first IPADDR
# IPADDR=$(ifconfig -a | awk '/inet /{printf (split($2,a,":")>1)?a[2]:$2" "}')
if [ x"${HACTYPE}" = "xPRI" -o x"${HACTYPE}" = "xSTD" ]; then 
  _loginfo "IPADDR: ${IP_PRI}"
  GBASESERVER=${SN_PRI}
elif [ x"${HACTYPE}" = "xSEC" ]; then
  _loginfo "IPADDR: ${IP_SEC}"
  GBASESERVER=${SN_SEC}
else
  _loginfo "IPADDR: 0.0.0.0"
fi
_loginfo "Datadir: ${DATADIR}"

##### Get env
NUMCPU=$(awk '/^processor/{i++}END{printf("%d\n",i)}' /proc/cpuinfo)
NUMMEM=$(awk '/^MemTotal:/{printf("%d\n",$2/1000)}' /proc/meminfo)
if [ -x /usr/bin/lscpu -o -x /usr/sbin/lscpu ]; then 
  NUMNODE=$(lscpu | awk '/NUMA node\(s\)/{print $NF}')
fi
NUMNODE=${NUMNODE:-1}

if [ x"${USER_INST}" = "xsmall" ]; then
  NUMCPU=1
  NUMMEM=2048
  DBS1GB=y
else
  if [[ ${USER_CPU} -gt 0 && ${USER_CPU} -lt ${NUMCPU} ]]; then
    NUMCPU=${USER_CPU}
  fi
  if [[ ${USER_MEM} -gt 0 && ${USER_MEM} -lt ${NUMMEM} ]]; then
    NUMMEM=${USER_MEM}
  fi
fi

### dbspace init size.
# default chunk size, 16G
ROOTSIZE=1024000
PLOGSIZE=2048000
LLOGSIZE=4096000
SBSPACESIZE=4096000
TEMPSIZE=4096000
DATASIZE=10240000
DMAXSIZE=2048000000
ERSBSSIZE=1024000
ERDBSSIZE=1024000

if [[ x"$DBS1GB" = xy || x"$DBS1GB" = xY ]]; then
  ROOTSIZE=1024000
  PLOGSIZE=1024000
  LLOGSIZE=1024000
  SBSPACESIZE=1024000
  TEMPSIZE=1024000
  DATASIZE=1024000
fi
METADATASIZE=$(expr ${SBSPACESIZE} / 5)
if [ ${METADATASIZE:-0} -eq 0 ]; then
  METADATAINFO=""
else
  METADATAINFO=" -Mo 0 -Ms ${METADATASIZE}"
fi

if [ ${NUMCPU:-0} -eq 0 ]; then
  echo "GET cpu information error."
  exit 2 
elif [ $NUMCPU -le 4 ]; then
  CPUVP=$NUMCPU
  CFG_NETPOOL=1
else
  if [ ${NUMNODE} -gt 2 ]; then 
    CPUVP=$(expr ${NUMCPU} \* 2 / ${NUMNODE})
  else 
    CPUVP=$(expr ${NUMCPU} - 1)
  fi
  CFG_NETPOOL=$(expr ${NUMCPU} / 3)
  [ ${CFG_NETPOOL} -gt 10 ] && CFG_NETPOOL=10
fi

if [ ${NUMMEM:-0} -eq 0 ]; then
  echo "GET memory information error."
  exit 2
elif [ $NUMMEM -le 2048 ]; then
  # mem less then 2G
  CFG_LOCKS=50000
  CFG_SHMVIRTSIZE=384000
  CFG_2KPOOL=20000
  CFG_16KPOOL=10000 
elif [ $NUMMEM -le 4096 ]; then
  # mem less then 4G
  CFG_LOCKS=200000
  CFG_SHMVIRTSIZE=512000
  CFG_2KPOOL=50000
  CFG_16KPOOL=20000 
elif [ $NUMMEM -le 8192 ]; then
  # mem less then 8G
  CFG_LOCKS=1000000
  CFG_SHMVIRTSIZE=512000
  CFG_2KPOOL=500000
  CFG_16KPOOL=100000
elif [ $NUMMEM -le 32768 ]; then
  # mem >8G && < 32G, use 2k & 16k buffpool
  MUTI=$(expr $NUMMEM / 8000)
  [ $MUTI -eq 0 ] && MUTI=2
  CFG_LOCKS=5000000
  CFG_SHMVIRTSIZE=$(echo "$MUTI" | awk '{printf("%d\n",($1 - 1) * 1024000)}')
  CFG_2KPOOL=500000
  CFG_16KPOOL=$(echo "$MUTI" | awk '{printf("%d\n",($1 - 1) * 200000)}')
else
  # mem > 32G
  CFG_LOCKS=5000000
  CFG_SHMVIRTSIZE=4096000
  CFG_2KPOOL=1000000
  CFG_16KPOOL=1000000
fi
PAGESIZE="-k 16"
CFG_SHMADD=$(expr ${CFG_SHMVIRTSIZE:-1024000} / 4)
CFG_SHMTOTAL=$(expr $NUMMEM \* 900)

##### Create group and user
id ${USER_NAME} >/dev/null 2>&1
if [ $? -eq 0 ];then
  echo "User: ${USER_NAME} exists, you must check !" 
  exit 2
fi
_loginfo "Creating group [${USER_NAME}] and user [${USER_NAME}] with HOME [$USER_HOME]."
groupadd ${USER_NAME} 2>/dev/null
if [ $? -gt 0 ]; then
  echo "Create group [${USER_NAME}] error."
  exit 3
fi
TMPUSER_HOME=${USER_HOME%/*}
if [ ! -d ${TMPUSER_HOME} ]; then
  _loginfo "Creating directory where user home will build: ${TMPUSER_HOME}" 
  mkdir -p ${TMPUSER_HOME}
fi
useradd -g ${USER_NAME} -d ${USER_HOME:-/home/${USER_NAME}} -m -s /bin/bash ${USER_NAME} 2>/dev/null
if [ $? -gt 0 ]; then
  echo "Create user [${USER_NAME}] error."
  exit 3
fi
echo "${USER_NAME}:${USER_PASS}" | chpasswd 2>/dev/null
mkdir -p ${USER_HOME}/users 2>/dev/null
chmod 0755 ${USER_HOME} 2>/dev/null
chown ${USER_NAME}:${USER_NAME} ${USER_HOME} 2>/dev/null
chmod 0777 ${USER_HOME}/users 2>/dev/null
chown ${USER_NAME}:${USER_NAME} ${USER_HOME}/users 2>/dev/null
if [ ${ERTYPE:-0} -gt 0 ]; then
  mkdir -p ${USER_HOME}/atsdir 2>/dev/null
  chown ${USER_NAME}:${USER_NAME} ${USER_HOME}/atsdir 2>/dev/null
  mkdir -p ${USER_HOME}/risdir 2>/dev/null
  chown ${USER_NAME}:${USER_NAME} ${USER_HOME}/risdir 2>/dev/null
fi

##### Unzip software and install
_loginfo "Unziping [${SOFTPACKNAME}]."
mkdir -p ${WORKDIR}/install 2>/dev/null
cd ${WORKDIR}/install
tar -xf ${WORKDIR}/${SOFTPACKNAME} 2>/dev/null
if [ ! -x "ids_install" ]; then
  echo "ids_install not exists."
  exit 4
fi
if [ -x "onsecurity" ]; then
  _loginfo "Check path INSTALL_DIR(${INSTALL_DIR}) security."
  ${WORKDIR}/install/onsecurity ${INSTALL_DIR} >/dev/null 2>&1
  if [ $? -gt 0 ]; then
    echo "INSTALL_DIR: ${INSTALL_DIR} not security, Plase Check."
    exit 5
  fi
fi

mkdir -p ${INSTALL_DIR} 2>/dev/null
chown ${USER_NAME}:${USER_NAME} ${INSTALL_DIR} 2>/dev/null
chmod 0755 ${INSTALL_DIR} 2>/dev/null

# ids_install
mkdir /${USER_UPPER}TMP 2>/dev/null
chown ${USER_NAME}:${USER_NAME} /${USER_UPPER}TMP 2>/dev/null
chmod 1775 /${USER_UPPER}TMP 2>/dev/null
mkdir -p ${WORKDIR}/tmp 2>/dev/null
export IATEMPDIR=${WORKDIR}/tmp
export LD_LIBRARY_PATH=/usr/lib64:/usr/lib:/lib64:/lib:/lib/x86_64-linux-gnu:/lib/aarch64-linux-gnu:$LD_LIBRARY_PATH
_loginfo "Execute software install, this will take a moment."
timeout 1800 ${WORKDIR}/install/ids_install -i silent -DLICENSE_ACCEPTED=TRUE -DUSER_INSTALL_DIR=${INSTALL_DIR:-/opt/${USER_NAME}}
if [ ! $? -eq 0 ]; then
  _loginfo "Install Error! remove install file."
  if [ -d ${INSTALL_DIR:-/opt/${USER_NAME}} ]; then
    rm -rf ${INSTALL_DIR:-/opt/${USER_NAME}}
  fi
  userdel -r ${USER_NAME} 2>/dev/null
  groupdel ${USER_NAME} 2>/dev/null
  exit 6
fi

if [ -f ${INSTALL_DIR}/etc/alarmprogram.sh ]; then
  sed -i "s#/ksh#/sh#g" ${INSTALL_DIR}/etc/alarmprogram.sh
fi
if [ -f ${INSTALL_DIR}/bin/onshutdown.sh ]; then
  sed -i "s#/ksh#/sh#g" ${INSTALL_DIR}/bin/onshutdown.sh
fi

###### Init database
[ ! -d /etc/${USER_NAME} ] && mkdir -p /etc/${USER_NAME} 2>/dev/null
echo "USERS:daemon" > /etc/${USER_NAME}/allowed.surrogates

# profile
_loginfo "Building ~${USER_NAME}/.bash_profile ."
cat >> $USER_HOME/.bash_profile <<EOF 2>/dev/null
export ${USER_UPPER}DIR=${INSTALL_DIR}
export ${USER_UPPER}SERVER=${GBASESERVER}
export ONCONFIG=onconfig.\$${USER_UPPER}SERVER
export PATH=\$${USER_UPPER}DIR/bin:\$${USER_UPPER}DIR/sbin:\${PATH}
export ${USER_UPPER}SQLHOSTS=${INSTALL_DIR}/etc/sqlhosts
export SERVER_LOCALE=${GBASELOCALE}
export DB_LOCALE=${GBASELOCALE}
export CLIENT_LOCALE=${GBASELOCALE}
export GL_USEGLU=1
# export NODEFDAC=yes
export DBDATE="Y4MD-"
export DBACCESS_SHOW_TIME=1
# export IFX_NO_TIMELIMIT_WARNING=y
EOF
chown ${USER_NAME}:${USER_NAME} $USER_HOME/.bash_profile
chmod 0644 $USER_HOME/.bash_profile

# sqlhosts
_loginfo "Building ${INSTALL_DIR}/etc/sqlhosts ."
# echo "$GBASESERVER onsoctcp 0.0.0.0 ${PORTNO:-9088}" > ${INSTALL_DIR}/etc/sqlhosts
if [ ${ERTYPE:-0} -eq 2 -o ${ERTYPE:-0} -eq 12 ]; then
  GID=2
else
  GID=1
fi
if [ x"${HACTYPE}" = "xSTD" ]; then
  cat > ${INSTALL_DIR}/etc/sqlhosts <<EOF 2>/dev/null
# group
g_${SN_PRI}        group    -       -   i=${GID:-1},e=${SN_PRI}
${SN_PRI}          onsoctcp    ${IP_PRI}  ${PORTNO}  g=g_${SN_PRI}

# priv
gbase01    onsoctcp    0.0.0.0    ${PORTPRIV}
EOF
else
  cat > ${INSTALL_DIR}/etc/sqlhosts <<EOF 2>/dev/null
# group
g_${SN_PRI}_${SN_SEC}  group       -          -          i=${GID:-1},e=${SN_SEC}
${SN_PRI}                onsoctcp    ${IP_PRI}  ${PORTNO}  g=g_${SN_PRI}_${SN_SEC}
${SN_SEC}                onsoctcp    ${IP_SEC}  ${PORTNO}  g=g_${SN_PRI}_${SN_SEC}

# priv
gbase01    onsoctcp    0.0.0.0    ${PORTPRIV}
EOF
fi
chown ${USER_NAME}:${USER_NAME} ${INSTALL_DIR}/etc/sqlhosts
chmod 0644 ${INSTALL_DIR}/etc/sqlhosts

# host.trust
if [ ${ERTYPE:-0} -gt 0 -o x"${HACTYPE}" = "xPRI" -o x"${HACTYPE}" = "xSEC" ]; then
  _loginfo "Building ${INSTALL_DIR}/etc/host.trust ."
  echo "+ ${USER_NAME}" > ${INSTALL_DIR}/etc/host.trust 2>/dev/null
  chown ${USER_NAME}:${USER_NAME} ${INSTALL_DIR}/etc/host.trust
  chmod 0644 ${INSTALL_DIR}/etc/host.trust
fi

# onconfig
CFGFILE=${INSTALL_DIR}/etc/onconfig.$GBASESERVER
cp ${INSTALL_DIR}/etc/onconfig.std ${CFGFILE}
chown ${USER_NAME}:${USER_NAME} ${CFGFILE}
chmod 0644 ${CFGFILE}
_loginfo "Building ${CFGFILE} ."

sed -i "s#^ROOTPATH.*#ROOTPATH ${DATADIR}/rootchk#g" ${CFGFILE}
sed -i "s#^ROOTSIZE.*#ROOTSIZE $ROOTSIZE#g" ${CFGFILE}
sed -i "s#^DBSERVERNAME.*#DBSERVERNAME $GBASESERVER#g" ${CFGFILE}
# sed -i "s#^DBSERVERALIASES.*#DBSERVERALIASES ${GBASESERVER}_ssl#g" ${CFGFILE}
sed -i "s#^TAPEDEV.*#TAPEDEV /dev/null#g" ${CFGFILE}
sed -i "s#^LTAPEDEV.*#LTAPEDEV /dev/null#g" ${CFGFILE}
sed -i "s#^USERMAPPING.*#USERMAPPING ADMIN#g" ${CFGFILE}
sed -i "s#^DEF_TABLE_LOCKMODE.*#DEF_TABLE_LOCKMODE row#g" ${CFGFILE}

# datadir
_loginfo "Creating DATADIR: ${DATADIR} ."
mkdir -p ${DATADIR}
_loginfo "Change permission for directory: ${DATADIR} ."
chown ${USER_NAME}:${USER_NAME} ${DATADIR}

DTDIR=${DATADIR}
while true
do
  DTTMP="${DTDIR%/*}"
  DTDIR=$DTTMP
  if [ x"${DTDIR}" = "x" ]; then
    break
  else
    _loginfo "Change permission for directory: $DTDIR ."
    chmod 0755 $DTDIR
  fi
  sleep 1
done

TMPDIR=$(pwd)
cd ${DATADIR}
> rootchk
touch plogchk llogchk sbspace01 tempchk01 datachk01
chown ${USER_NAME}:${USER_NAME} rootchk plogchk llogchk sbspace01 tempchk01 datachk01
chmod 0660 rootchk plogchk llogchk sbspace01 tempchk01 datachk01
if [ ${ERTYPE:-0} -gt 0 ]; then
  touch erdbs01 ersbs01
  chown ${USER_NAME}:${USER_NAME} erdbs01 ersbs01
  chmod 0660 erdbs01 ersbs01
fi
cd $TMPDIR

if [ "x$SOFTONLY" = xy -o "x$SOFTONLY" = xY ];then
  _loginfo "GBase 8s Database software install finished."
  cat <<!

  You MUST manually initialize the database!

!
  exit 0;
fi

# fix 330 ddlalert error
_loginfo "Creating file ${INSTALL_DIR}/etc/upgraded ."
if [ ! -f ${INSTALL_DIR}/etc/upgraded ]; then
  touch ${INSTALL_DIR}/etc/upgraded
  chown ${USER_NAME}:${USER_NAME} ${INSTALL_DIR}/etc/upgraded
  chmod 0644 ${INSTALL_DIR}/etc/upgraded
fi

# oninit
_loginfo "Start run database init: oninit -ivy"
su - ${USER_NAME} -c "timeout 1800 oninit -ivy"
if [ ! $? -eq 0 ]; then
  _loginfo "Database init error! Please check online.log."
  exit 1
fi

echo -e "OK"

# check onstat -g ntt
_loginfo "Checking status for network listener."
ISNTTON=$(su - ${USER_NAME} -c 'onstat -g ntt | grep "soctcplst.*soctcp" | wc -l')
if [ ${ISNTTON} -eq 0 ]; then
  _loginfo "Network listener ERROR ! Please check online.log for more information."
  exit 1
fi 

_loginfo "Creating system database.\c"
for w in {1..5}
do
  sleep 3
  echo -e ".\c"
done

# check system database build.
NUMTIMES=1
while :
do
  NUMDB=$(su - ${USER_NAME} -c 'echo "output to pipe cat without headings select count(*) numdb from sysdatabases;" | dbaccess sysmaster - 2>/dev/null')
  if [ ${NUMDB:-0} -gt 3 ]; then
    break
  else
    sleep 3
    echo -e ".\c"
  fi
  if [ ${NUMTIMES:-1} -gt 100 ]; then
    _loginfo "Create system database ERROR ! Please check online.log for more information."
    exit 1
  fi
  NUMTIMES=$(expr ${NUMTIMES} + 1) 
done

## create dbspace.
echo -e ""
_loginfo "Creating dbspace plogdbs."
su - ${USER_NAME} -c "onspaces -c -d plogdbs -p ${DATADIR}/plogchk -o 0 -s $PLOGSIZE >/dev/null 2>&1"

_loginfo "Creating dbspace llogdbs."
su - ${USER_NAME} -c "onspaces -c -d llogdbs -p ${DATADIR}/llogchk -o 0 -s $LLOGSIZE >/dev/null 2>&1"

_loginfo "Creating smart blob space sbspace01"
su - ${USER_NAME} -c "onspaces -c -S sbspace01 -p ${DATADIR}/sbspace01 -o 0 -s $SBSPACESIZE $METADATAINFO >/dev/null 2>&1"

_loginfo "Creating dbspace tempdbs01"
su - ${USER_NAME} -c "onspaces -c -d tempdbs01 -p ${DATADIR}/tempchk01 -t -o 0 -s $TEMPSIZE $PAGESIZE >/dev/null 2>&1"

_loginfo "Creating dbspace datadbs01"
su - ${USER_NAME} -c "onspaces -c -d datadbs01 -p ${DATADIR}/datachk01 -o 0 -s $DATASIZE $PAGESIZE >/dev/null 2>&1"

if [ ${ERTYPE:-0} -gt 0 ]; then
  _loginfo "Creating dbspace erdbs01"
  su - ${USER_NAME} -c "onspaces -c -d erdbs01 -p ${DATADIR}/erdbs01 -o 0 -s $ERDBSSIZE $PAGESIZE >/dev/null 2>&1"

  _loginfo "Creating dbspace ersbs01"
  su - ${USER_NAME} -c "onspaces -c -S ersbs01 -p ${DATADIR}/ersbs01 -o 0 -s $ERSBSSIZE >/dev/null 2>&1"
fi
## change chunk extend able on
_loginfo "Changing auto extend able on for chunk datadbs01"
ADMIN_SQLFILE=${INSTALL_DIR}/temp/admin_sqlfile.sql
mkdir -p ${INSTALL_DIR}/temp
cat << ! > $ADMIN_SQLFILE 2>&1
-- set datadbs01 extendable
EXECUTE FUNCTION task ('modify chunk extendable on', 6);
EXECUTE FUNCTION task ('modify space sp_sizes','datadbs01','10','100000','${DMAXSIZE}');
-- set all task not run except idle_user_timeout
update ph_task set tk_enable = 'f';
update ph_task set tk_frequency = interval(00:30:00) hour to second, tk_enable = 't' 
where tk_name = 'idle_user_timeout';
-- procedure
drop procedure if exists delete_task_logs;
create procedure delete_task_logs()
  define v_datestr datetime year to second;
  select current year to second - value::interval day to second
  into v_datestr
  from ph_threshold
  where task_name = 'delete_task_logs';
  if v_datestr is not null AND v_datestr <> '' then
    delete from command_history where cmd_exec_time < v_datestr;
    delete from ph_run where run_time < v_datestr;
    -- delete from ph_alerts where alert_time < v_datestr;
  end if;
end procedure;
-- add task delete_task_logs into ph_threshold
delete from ph_threshold where task_name = 'delete_task_logs';
insert into ph_threshold (id, name, task_name, value, value_type, description)
values(0,'DELETE ALL SYSADMIN LOGS','delete_task_logs','1 00:00:00','STRING','Delete command_history and ph_run logs');
-- add task delete_task_logs into ph_task
delete from ph_task where tk_name = 'delete_task_logs';
insert into ph_task( tk_name, tk_description, tk_type, tk_dbs, tk_execute, tk_start_time, tk_stop_time, tk_frequency, tk_group, tk_enable)
values('delete_task_logs','Delete command_history and ph_run logs','TASK','sysadmin','execute procedure delete_task_logs()',
       DATETIME(00:05:00) hour to second , DATETIME(01:00:00) hour to second, INTERVAL ( 1 ) DAY TO DAY, 'MISC','t');
!
if [ -s $ADMIN_SQLFILE ]; then
  chown ${USER_NAME}:${USER_NAME} ${INSTALL_DIR}/temp
  chown ${USER_NAME}:${USER_NAME} $ADMIN_SQLFILE
  su - ${USER_NAME} -c "dbaccess sysadmin $ADMIN_SQLFILE >/dev/null 2>&1"
fi

## create mapping default user
_loginfo "Creating default user for mapping user"
ADMIN_SQLFILE=${INSTALL_DIR}/temp/mapping_sqlfile.sql
mkdir -p ${INSTALL_DIR}/temp
cat << ! > $ADMIN_SQLFILE 2>&1
CREATE DEFAULT USER WITH PROPERTIES USER daemon HOME "${USER_HOME}/users";
create user dbtuser with password "$USER_PASS";
!
if [ -s $ADMIN_SQLFILE ]; then
  chown ${USER_NAME}:${USER_NAME} ${INSTALL_DIR}/temp
  chown ${USER_NAME}:${USER_NAME} $ADMIN_SQLFILE
  su - ${USER_NAME} -c "dbaccess sysuser $ADMIN_SQLFILE >/dev/null 2>&1"
fi

## physical log & logical log
_loginfo "Moving physical log to plogdbs."
PLOGFILE=$(echo $PLOGSIZE | awk '{printf("%d\n",substr($1,1,1) * 10 ^ (length($1) - 1))}')
su - ${USER_NAME} -c "onparams -p -d plogdbs -s $PLOGFILE -y >/dev/null 2>&1"

_loginfo "Adding 10 logical log file in llogdbs."
LLOGFILE=$(echo $LLOGSIZE | awk '{printf("%d\n",substr($1,1,1) * 10 ^ (length($1) - 1))}')
NEWFILE=$(expr $LLOGFILE / 10)
[ $NEWFILE -gt 1000000 ] && NEWFILE=1000000
for w in {1..10}
do
  su - ${USER_NAME} -c "onparams -a -d llogdbs -s $NEWFILE >/dev/null 2>&1"
done

_loginfo "Moving CURRENT logical log to new logical file."
while :
do
  CURRLOG=$(su - ${USER_NAME} -c "onmode -l; onmode -c; onstat -l" | awk '/U---C-L/{print $2}')
  if [ $CURRLOG -gt 6 ]; then
    break
  else
    sleep 1
  fi
done

_loginfo "Droping logical log file which in rootdbs."
for e in {1..6}
do
  su - ${USER_NAME} -c "onparams -d -l $e -y >/dev/null 2>&1"
done

### Build default database: testdb
_loginfo "Create database testdb."
CRDB_SQLFILE=${INSTALL_DIR}/temp/crdb_sqlfile.sql
mkdir -p ${INSTALL_DIR}/temp
cat << ! > $CRDB_SQLFILE 2>&1
create database testdb in datadbs01 with log;
grant dba to dbtuser;
!
if [ -s $CRDB_SQLFILE ]; then
  chown ${USER_NAME}:${USER_NAME} ${INSTALL_DIR}/temp
  chown ${USER_NAME}:${USER_NAME} $CRDB_SQLFILE
  su - ${USER_NAME} -c "dbaccess sysmaster $CRDB_SQLFILE >/dev/null 2>&1"
fi

###### change config
#_loginfo "Creating file \${INSTALL_DIR}/etc/sysadmin/stop ."
#if [ ! -f ${INSTALL_DIR}/etc/sysadmin/stop ]; then
#  touch ${INSTALL_DIR}/etc/sysadmin/stop
#  chown ${USER_NAME}:${USER_NAME} ${INSTALL_DIR}/etc/sysadmin/stop
#  chmod 0644 ${INSTALL_DIR}/etc/sysadmin/stop
#fi

## cfg with static value
_loginfo "Optimizing database config."
sed -i "s#^PHYSBUFF.*#PHYSBUFF 1024#g" ${CFGFILE}
sed -i "s#^LOGBUFF.*#LOGBUFF 1024#g" ${CFGFILE}
sed -i "s#^DYNAMIC_LOGS.*#DYNAMIC_LOGS 0#g" ${CFGFILE}
sed -i "s#^LTXHWM.*#LTXHWM 40#g" ${CFGFILE}
sed -i "s#^LTXEHWM.*#LTXEHWM 50#g" ${CFGFILE}
sed -i "s#^DBSPACETEMP.*#DBSPACETEMP tempdbs01#g" ${CFGFILE}
sed -i "s#^SBSPACENAME.*#SBSPACENAME sbspace01#g" ${CFGFILE}
sed -i "s#^SYSSBSPACENAME.*#SYSSBSPACENAME sbspace01#g" ${CFGFILE}
sed -i "s#^MULTIPROCESSOR.*#MULTIPROCESSOR 1#g" ${CFGFILE}
sed -i "s#^AUTO_TUNE.*#AUTO_TUNE 0#g" ${CFGFILE}
sed -i "/^AUTO_TUNE/a\AUTO_REPREPARE 1" ${CFGFILE}
sed -i "/^AUTO_TUNE/a\ENABLE_NULL_STRING  0" ${CFGFILE}
sed -i "s#^CLEANERS.*#CLEANERS 128#g" ${CFGFILE}
sed -i "s#^STACKSIZE.*#STACKSIZE 1024#g" ${CFGFILE}
sed -i "s#^ALLOW_NEWLINE.*#ALLOW_NEWLINE 1#g" ${CFGFILE}
sed -i "s#^DS_MAX_QUERIES.*#DS_MAX_QUERIES 5#g" ${CFGFILE}
sed -i "s#^DS_TOTAL_MEMORY.*#DS_TOTAL_MEMORY 1024000#g" ${CFGFILE}
sed -i "s#^DS_NONPDQ_QUERY_MEM.*#DS_NONPDQ_QUERY_MEM 256000#g" ${CFGFILE}
sed -i "s#^TEMPTAB_NOLOG.*#TEMPTAB_NOLOG 1#g" ${CFGFILE}
sed -i "s#^DUMPSHMEM.*#DUMPSHMEM 0#g" ${CFGFILE}
# sed -i "s#^IFX_FOLDVIEW.*#IFX_FOLDVIEW 0#g" ${CFGFILE}
# sed -i "s#^OPT_GOAL.*#OPT_GOAL 0#g" ${CFGFILE}
sed -i "s#^TAPEBLK.*#TAPEBLK 512#g" ${CFGFILE}
sed -i "s#^LTAPEBLK.*#LTAPEBLK 512#g" ${CFGFILE}
sed -i "s#^ALARMPROGRAM.*#ALARMPROGRAM \$GBASEDBTDIR/etc/log_full.sh#g" ${CFGFILE}
sed -i "s#^MAX_FILL_DATA_PAGES.*#MAX_FILL_DATA_PAGES 1#g" ${CFGFILE}

if [ $NUMMEM -le 4096 ]; then
  sed -i "s#^DS_TOTAL_MEMORY.*#DS_TOTAL_MEMORY 128000#g" ${CFGFILE}
  sed -i "s#^DS_NONPDQ_QUERY_MEM.*#DS_NONPDQ_QUERY_MEM 32000#g" ${CFGFILE}
fi
# dynamic value
sed -i "s#^NETTYPE.*#NETTYPE soctcp,${CFG_NETPOOL},200,NET#g" ${CFGFILE}
sed -i "s#^VPCLASS.*cpu.*#VPCLASS cpu,num=${CPUVP},noage#g" ${CFGFILE}
sed -i "s#^LOCKS.*#LOCKS ${CFG_LOCKS}#g" ${CFGFILE}
sed -i "s#^SHMVIRTSIZE.*#SHMVIRTSIZE ${CFG_SHMVIRTSIZE}#g" ${CFGFILE}
sed -i "s#^SHMADD.*#SHMADD ${CFG_SHMADD}#g" ${CFGFILE}
sed -i "s#^EXTSHMADD.*#EXTSHMADD ${CFG_SHMADD}#g" ${CFGFILE}
sed -i "s#^SHMTOTAL.*#SHMTOTAL ${CFG_SHMTOTAL}#g" ${CFGFILE}

sed -i "s#^BUFFERPOOL.*size=2.*#BUFFERPOOL size=2K,buffers=${CFG_2KPOOL},lrus=8,lru_min_dirty=50,lru_max_dirty=60#g" ${CFGFILE}
sed -i "s#^BUFFERPOOL.*size=16.*#BUFFERPOOL size=16K,buffers=${CFG_16KPOOL},lrus=64,lru_min_dirty=50,lru_max_dirty=60#g" ${CFGFILE}

MSG_DATE=$(su - ${USER_NAME} -c "onstat -c MSG_DATE 2>/dev/null")
if [ x"${MSG_DATE}" = x -a "x${USER_NAME}" = "xgbasedbt" ]; then
  sed -i "/^MSGPATH/a\MSG_DATE 1" ${CFGFILE}
fi

sed -i "s#^DBSERVERALIASES.*#DBSERVERALIASES gbase01#g" ${CFGFILE}
if [ ${ERTYPE:-0} -gt 0 ]; then
  sed -i "s#^CDR_QDATA_SBSPACE.*#CDR_QDATA_SBSPACE ersbs01#g" ${CFGFILE}
  sed -i "s#^CDR_DBSPACE.*#CDR_DBSPACE erdbs01#g" ${CFGFILE}
  sed -i "s#^CDR_QHDR_DBSPACE.*#CDR_QHDR_DBSPACE erdbs01#g" ${CFGFILE}
  sed -i "s#^REMOTE_SERVER_CFG.*#REMOTE_SERVER_CFG host.trust#g" ${CFGFILE}
fi

if [ x"${HACTYPE}" = "xPRI" -o x"${HACTYPE}" = "xSEC" ]; then
  sed -i "s#^HA_ALIAS.*#HA_ALIAS ${GBASESERVER}#g" ${CFGFILE}
  sed -i "s#^DRAUTO.*#DRAUTO 2#g" ${CFGFILE}
  sed -i "s#^DRINTERVAL.*#DRINTERVAL 10#g" ${CFGFILE}
  sed -i "s#^VRRP_CONF_PATH.*#VRRP_CONF_PATH ${INSTALL_DIR}/etc/vrrp.conf#g" ${CFGFILE}
  sed -i "s#^REMOTE_SERVER_CFG.*#REMOTE_SERVER_CFG host.trust#g" ${CFGFILE}
fi
if [ ${ERTYPE:-0} -eq 1 -o ${ERTYPE:-0} -eq 11 ]; then
  sed -i "s#^CDR_SERIAL.*#CDR_SERIAL 2,1#g" ${CFGFILE}
elif [ ${ERTYPE:-0} -eq 2 -o ${ERTYPE:-0} -eq 12 ]; then
  sed -i "s#^CDR_SERIAL.*#CDR_SERIAL 2,0#g" ${CFGFILE}
fi

chown ${USER_NAME}:${USER_NAME} ${CFGFILE}

### set audit 
if [[ x"${SECNEED}" = xy || x"${SECNEED}" = xY ]]; then
  _loginfo "Create group dbaao and user dbaao ."
  groupadd dbaao 2>/dev/null
  if [ $? -gt 0 ]; then 
    _loginfo "Create group dbaao error !"
  fi
  useradd -g dbaao -d ${TMPUSER_HOME}/dbaao -m -s /bin/bash dbaao 2>/dev/null
  if [ $? -gt 0 ]; then 
    _loginfo "Create user dbaao error !"
  fi

  _loginfo "Create group dbsso and user dbsso ."
  groupadd dbsso 2>/dev/null
  if [ $? -gt 0 ]; then 
    _loginfo "Create group dbsso error !"
  fi
  useradd -g dbsso -d ${TMPUSER_HOME}/dbsso -m -s /bin/bash dbsso 2>/dev/null
  if [ $? -gt 0 ]; then 
    _loginfo "Create user dbsso error !"
  fi

  _loginfo "Change permission for user directory."
  chgrp -R dbaao ${INSTALL_DIR}/aaodir 2>/dev/null
  chgrp -R dbsso ${INSTALL_DIR}/dbssodir 2>/dev/null
  echo "IXUSERS=*" > ${INSTALL_DIR}/dbssodir/seccfg
  chown ${USER_NAME}:dbsso ${INSTALL_DIR}/dbssodir/seccfg 2>/dev/null

  mkdir -p ${TMPUSER_HOME}/dbaao/aaodir 2>/dev/null
  chmod 0755 ${TMPUSER_HOME}/dbaao/aaodir 2>/dev/null
  chown dbaao:dbaao ${TMPUSER_HOME}/dbaao/aaodir 2>/dev/null
  chmod 0750 ${TMPUSER_HOME}/dbaao 2>/dev/null
  chmod 0750 ${TMPUSER_HOME}/dbsso 2>/dev/null  

  cp ${USER_HOME}/.bash_profile ${TMPUSER_HOME}/dbaao/.bash_profile 2>/dev/null
  chown dbaao:dbaao ${TMPUSER_HOME}/dbaao/.bash_profile 2>/dev/null
  cp ${USER_HOME}/.bash_profile ${TMPUSER_HOME}/dbsso/.bash_profile 2>/dev/null
  chown dbsso:dbsso ${TMPUSER_HOME}/dbaao/.bash_profile 2>/dev/null

  _loginfo "Set audit on."  
  sed -i "s#^ADTMODE.*#ADTMODE 1#g" ${INSTALL_DIR}/aaodir/adtcfg
  sed -i "s#/usr/${USER_NAME}/aaodir#${TMPUSER_HOME}/dbaao/aaodir#g" ${INSTALL_DIR}/aaodir/adtcfg

  _loginfo "Set trust file."
  sed -i "s#^REMOTE_SERVER_CFG.*#REMOTE_SERVER_CFG host.trust#g" ${CFGFILE}
  echo "+ gbasedbt" > ${INSTALL_DIR}/etc/host.trust
  chown ${USER_NAME}:${USER_NAME} ${INSTALL_DIR}/etc/host.trust 2>/dev/null
  chmod 0644 ${INSTALL_DIR}/etc/host.trust 2>/dev/null
fi

####### restart database
_loginfo "Restart GBase 8s Database Server."
su - ${USER_NAME} -c "timeout 1800 onmode -ky"
sleep 5
su - ${USER_NAME} -c "oninit -vy"
if [ $? -eq 0 ]; then
  if [ x"$SECNEED" = xy -o x"$SECNEED" = xY ]; then
    _loginfo "Set audit mask."
    su - dbsso -c "onaudit -a -u _default -e +DRDB,DRTB,ALTB,TCTB,STSN" 2>/dev/null
  fi
  _loginfo "Finish."
else
  _loginfo "Restart Server with ERROR. Please reduce the memory parameters(SHMVIRTSIZE,BUFFPOOLS), and then restart the server."
fi

### addition shell script: dbt
cat <<SHELL > ${INSTALL_DIR}/bin/dbt
#!/bin/bash

if [ ! \$# -eq 2 ];then
  cat <<EOF

  Usage: dbt DBNAME [ SQL | SQLFILE ]

EOF
exit 1
fi
DB=\$1
SQL=\$2

if [ -f "\${SQL}" ]; then
  dbaccess \${DB} \${SQL}
elif [ "x\${SQL}" = "x-" ]; then
  dbaccess \${DB} -
else
  dbaccess \${DB} << !SQL
\${SQL}
!SQL
fi
exit \$?
SHELL

if [ -f ${INSTALL_DIR}/bin/dbt ]; then
  chown ${USER_NAME}:${USER_NAME} ${INSTALL_DIR}/bin/dbt 2>/dev/null
  chmod 755 ${INSTALL_DIR}/bin/dbt 2>/dev/null
fi

### addition shell script: gbasecli
cat <<SHELL > ${INSTALL_DIR}/bin/gbasecli
#!/bin/bash
# filename : gbasecli
# function : build .netrc and invoke dbaccess connect
# write by : liaosnet@gbasedbt.com  2025-07-31

Usage(){
  cat <<EOF

  gbasecli 

      [-hHost] [-PPort] [-uUser] [-pPassword]
      [-sServername] [-dDbname] [-lLocale] [-gUseGLU] [-DDelimident]
      [-f]

      Servername: default 'gbase01'
      User      : default 'gbasedbt'
      Password  : default 'GBase123'  
      Host      : default '0.0.0.0'
      Port      : default '9088'
      Dbname    : default 'testdb'
      Locale    : default 'zh_CN.utf8'
      UseGLU    : default '0', accept '1','0'
      Delimident: default '0', accept '1','0'
      CleanCache: default false

EOF
  exit 1
}

if [ x"\${GBASEDBTDIR}" = x ]; then
  echo "\\\$GBASEDBTDIR not set !"
  exit 1
fi

while [[ \$# -gt 0 ]]
do
  key="\$1"
  case \${key:0:2} in
    -s) SERVERNAME="\${key:2}"; shift ;;
    -p) PASSWORD="\${key:2}";   shift ;;
    -u) USERNAME="\${key:2}";   shift ;;
    -P) PORT="\${key:2}";       shift ;;
    -h) HOST="\${key:2}";       shift ;;
    -d) DBNAME="\${key:2}";     shift ;;
    -l) LOCALE="\${key:2}";     shift ;;
    -g) USEGLU="\${key:2}";     shift ;;
    -D) DELIMIDENT="\${key:2}"; shift ;;
    -f) REBUILD="1";           shift ;;
    *)
        Usage
        ;;
  esac
done

SERVERNAME=\${SERVERNAME:-gbase01}
USERNAME=\${USERNAME:-gbasedbt}
PASSWORD=\${PASSWORD:-GBase123}
PORT=\${PORT:-9088}
HOST=\${HOST:-0.0.0.0}
DBNAME=\${DBNAME:-testdb}
USEGLU=\${USEGLU:-0}
LOCALE=\${LOCALE:-zh_CN.utf8}
DELIMIDENT=\${DELIMIDENT:-0}
REBUILD=\${REBUILD:-0}

# sqlhosts
if [ ! -f \${GBASEDBTDIR}/etc/sqlhosts.gbasecli ]; then
  cat <<EOF > \${GBASEDBTDIR}/etc/sqlhosts.gbasecli
\${SERVERNAME} onsoctcp \${HOST} \${PORT}
EOF
else
  ISEXISTS=\$(grep "^\${SERVERNAME}" \${GBASEDBTDIR}/etc/sqlhosts.gbasecli | wc -l)
  if [ \${ISEXISTS} -eq 0 ]; then
    cat <<EOF >> \${GBASEDBTDIR}/etc/sqlhosts.gbasecli
\${SERVERNAME} onsoctcp \${HOST} \${PORT}
EOF
  else
    ISEXISTS=\$(grep "^\${SERVERNAME}.*\${HOST}.*\${PORT}" \${GBASEDBTDIR}/etc/sqlhosts.gbasecli | wc -l)
    if [ \${ISEXISTS} -eq 0 ]; then
      sed -i "s#^\${SERVERNAME}.*#\${SERVERNAME} onsoctcp \${HOST} \${PORT}#g" \${GBASEDBTDIR}/etc/sqlhosts.gbasecli
    fi
  fi
fi

# .netrc
if [ ! -f \${HOME}/.netrc ]; then
  cat <<EOF > \${HOME}/.netrc
machine \${HOST} login \${USERNAME} password \${PASSWORD}
EOF
  chmod 600 \${HOME}/.netrc
else
  ISEXISTS=\$(grep "^machine.*\${HOST}" \${HOME}/.netrc | wc -l)
  if [ \${ISEXISTS} -eq 0 ]; then
    cat <<EOF >> \${HOME}/.netrc
machine \$HOST login \$USERNAME password \$PASSWORD
EOF
  else
    LASTMT=\$(stat -c %Y \${HOME}/.netrc)
    CURRTM=\$(date +%s)
    TIMEDIFF=\$(expr \${CURRTM} - \${LASTMT:-0})
    if [ x"\${REBUILD}" = x1 -o \${TIMEDIFF} -gt 300 ]; then
      sed -i "s#machine \${HOST}.*#machine \${HOST} login \${USERNAME} password \${PASSWORD}#g" \${HOME}/.netrc
    fi
  fi
fi

# dbaccess
export DB_LOCALE=\${LOCALE}
export CLIENT_LOCALE=\${LOCALE}
export GBASEDBTSQLHOSTS=\${GBASEDBTDIR}/etc/sqlhosts.gbasecli
if [ x"\${USEGLU}" = x0 ]; then
  unset GL_USEGLU
else
  export GL_USEGLU=1
fi
if [ x"\${DELIMIDENT}" = x0 ]; then
  unset DELIMIDENT
else
  export DELIMIDENT=y
fi

dbaccess \${DBNAME}@\${SERVERNAME} - 

SHELL

if [ -f ${INSTALL_DIR}/bin/gbasecli ]; then
  chown ${USER_NAME}:${USER_NAME} ${INSTALL_DIR}/bin/gbasecli 2>/dev/null
  chmod 755 ${INSTALL_DIR}/bin/gbasecli 2>/dev/null
fi

## output install information
cat <<EOF

--== GBase 8s Information for this install ==--
\$${USER_UPPER}SERVER : $GBASESERVER
\$${USER_UPPER}DIR    : ${INSTALL_DIR}
USER HOME       : $USER_HOME
DBSPACE DIR     : ${DATADIR}
IP ADDRESS      : ${IPADDR:-0.0.0.0}
PORT NUMBER     : $PORTNO
\$DB_LOCALE      : $GBASELOCALE
\$CLIENT_LOCALE  : $GBASELOCALE
JDBC URL        : jdbc:${USER_NAME}-sqli://IPADDR:$PORTNO/testdb:${USER_UPPER}SERVER=$GBASESERVER;DB_LOCALE=$GBASELOCALE;CLIENT_LOCALE=$GBASELOCALE;IFX_LOCK_MODE_WAIT=30;
JDBC USERNAME   : $USER_NAME
JDBC PASSWORD   : $USER_PASS
INNER USERNAME  : dbtuser
INNER PASSWORD  : $USER_PASS

EOF

exit 0
