#!/bin/bash
# filename: backup.sh
# function: GBase 8s Database Server backup shell.
# crontab(root)    : 0 1 * * * su - gbasedbt -c "/opt/gbase/script/backup.sh" >/dev/null 2>&1
# crontab(gbasedbt): 0 1 * * * . /home/gbase/.bash_profile && /opt/gbase/script/backup.sh >/dev/null 2>&1
# write  by: liaosnet@gbasedbt.com 2023-12-19
# update by: liaosnet@gbasedbt.com 2025-10-16
export LANG=C
WORKDIR=$(cd $(dirname $0) && pwd)
LOGFILE=${WORKDIR}/backup_status.log
[ ! -f ${LOGFILE} ] && touch ${LOGFILE}
loginfo(){
  echo -e "[$(date +'%Y-%m-%d %H:%M:%S')] $*" >> ${LOGFILE}
}

## define env
# USERNAME: gbasedbt OR informix
# 指定备份方式及日期，L0DAY表示0级备份日期1，4表示周1和周4，8/9表示无效日期
L0DAY=0
L1DAY=2,5
L2DAY=1,3,4,6
KEEPDAYS=7
USERNAME=gbasedbt
LOGNEED=1
CURRDAY=$(date +%w)

## check env
if [ ! x"$(whoami)" = "x${USERNAME}" ]; then
  loginfo "Must run as user: ${USERNAME}"
  exit 100
fi 

DBSTATUS=$(onstat -c|grep 'On-Line'|wc -l)
if [ ! "x${DBSTATUS}" = "x1" ]; then
  loginfo "Backup ONLY run On-Line Server which is Primary Server or Standard Server."
  exit 4
fi

# base on DBSTATUS 1 
BACKUPDIR=$(onstat -c TAPEDEV)
LOGSDIR=$(onstat -c LTAPEDEV)
HADBACKUP=$(onstat -g arc | awk '/^name/{need=1}need==1&&$2=="1"&&$3=="0"{print "1";exit}')

if [ ! -d ${BACKUPDIR} ]; then
  loginfo "The Backup dir is not exists."
  exit 3
fi

[ x"${LOGSDIR}" = "x/dev/null" ] && LOGNEED=0
if [[ ${LOGNEED} -eq 1 && ! -d "${LOGSDIR}" ]]; then
  loginfo "The logs backup dir is not exists."
  LOGNEED=0
fi

## backup db
# delete file
KEEPDAYS=${KEEPDAYS:-30}
loginfo "Delete data backup file(s) at $BACKUPDIR which out ${KEEPDAYS} days."
# function find had different action for path。
# or find . -name '*L[0-2]' -type f -mtime +${KEEPDAYS} -exec rm -rf {} \;
cd $BACKUPDIR && find . -name '*L[0-2]' -type f -mtime +${KEEPDAYS} | xargs rm -rf
if [ ${LOGNEED} -eq 1 ]; then
  loginfo "Delete log backup file(s) at $LOGSDIR which out ${KEEPDAYS} days."
  cd $LOGSDIR && find . -name '*_Log*' -type f -mtime +${KEEPDAYS} | xargs rm -rf
fi

if [ x"${HADBACKUP}" = x1 ]; then
  if [[ ${L0DAY} =~ ${CURRDAY} ]]; then
    loginfo "Execute L0 backup start..."
    ontape -s -L 0 >> ${LOGFILE} 2>&1
  elif [[ ${L1DAY} =~ ${CURRDAY} ]]; then
    loginfo "Execute L1 backup start..."
    ontape -s -L 1 >> ${LOGFILE} 2>&1
  elif [[ ${L2DAY} =~ ${CURRDAY} ]]; then
    loginfo "Execute L2 backup start..."
    ontape -s -L 2 >> ${LOGFILE} 2>&1
  fi
else
  loginfo "Execute FIRST L0 backup start..."
  ontape -s -L 0 >> ${LOGFILE} 2>&1
fi

BAKSTATUS=$?
if [ ${BAKSTATUS} -eq 0 ];then
  loginfo "Execute backup finished."
  loginfo "===== FINISH ============================================"
  exit 0
else
  loginfo "Execute backup error! Try Execute L0 backup start."
  ontape -s -L 0 >> ${LOGFILE} 2>&1
  BAKSTATUS=$?
  loginfo "Execute backup finished with CODE: ${BAKSTATUS}"
  loginfo "===== FINISH ============================================"
  exit ${BAKSTATUS}
fi
